"use client";

import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { AlertCircle, RefreshCw } from "lucide-react";
import { useAuth } from "../hooks/use-auth";
import SystemConfigurationCard from './SystemConfigurationCard';
import RoleManagement from './RoleManagement';
import PendingApprovals from './PendingApprovals';
import { useTranslation } from 'react-i18next';
import DeviceManagementTable from './DeviceManagementTable';
import DeviceDetailModal from './DeviceDetailModal';
import {
  DropdownMenu,
  DropdownMenuTrigger,
  DropdownMenuContent,
  DropdownMenuItem,
} from '../components/ui/dropdown-menu';
import PaymentHistoryTab from './PaymentHistoryTab';
import UserSettingsTab from './UserSettingsTab';
import PendingOrdersTab from './PendingOrdersTab';
import SystemStatsCard from './SystemStatsCard';

function useIsMobile(breakpoint = 768) {
  const [isMobile, setIsMobile] = useState(false);
  useEffect(() => {
    const check = () => setIsMobile(window.matchMedia(`(max-width: ${breakpoint}px)`).matches);
    check();
    window.addEventListener('resize', check);
    return () => window.removeEventListener('resize', check);
  }, [breakpoint]);
  return isMobile;
}

const SystemAdminTab: React.FC = () => {
  const { user, userPermissions, userRoleId } = useAuth();
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const isMobile = useIsMobile();
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'config' | 'push' | 'roles' | 'approvals' | 'devices' | 'payment-history' | 'user-settings' | 'pending-orders'>('config');

  // For push notification form
  const [users, setUsers] = useState<{ email: string; name?: string; f_name?: string; s_name?: string }[]>([]);
  const [selectedUser, setSelectedUser] = useState<string>('all');
  const [notifTitle, setNotifTitle] = useState('');
  const [notifMessage, setNotifMessage] = useState('');
  const [notifLoading, setNotifLoading] = useState(false);
  const [notifError, setNotifError] = useState<string | null>(null);
  const [notifSuccess, setNotifSuccess] = useState<string | null>(null);

  const [devices, setDevices] = useState<any[]>([]);
  const [search, setSearch] = useState('');
  const [selectedMobileDevice, setSelectedMobileDevice] = useState<string>('');
  const [selectedDevice, setSelectedDevice] = useState<any | null>(null);
  const [modalOpen, setModalOpen] = useState(false);
  const [searchUser, setSearchUser] = useState<string>('');

  // Расширенный маппинг типа устройства к ключу перевода
  const DEVICE_TYPE_MAP: Record<string, string> = {
    'waterMeter': 'devices.waterMeter',
    'electricMeter': 'devices.electricMeter',
    'Water meter': 'devices.waterMeter',
    'Electric meter': 'devices.electricMeter',
    'Water Meter': 'devices.waterMeter',
    'Electric Meter': 'devices.electricMeter',
    // Добавьте другие типы устройств по мере необходимости
  };

  const getDeviceTypeLabel = (type: string) => {
    if (!type) return '';
    const key = DEVICE_TYPE_MAP[type.trim()] || type.trim();
    return t(key, type);
  };

  useEffect(() => {
    if (activeTab === 'push') {
      fetch('/api/postgres/users')
        .then(res => res.json())
        .then(data => setUsers(data.users || []))
        .catch(() => setUsers([]));
    }
  }, [activeTab]);

  useEffect(() => {
    if (activeTab === 'config') {
      // Для пользователя с ролью 0 загружаем все устройства системы
      const endpoint = userRoleId === 0 ? '/api/postgres/all-devices' : '/api/postgres/all-user-devices';
      fetch(endpoint)
        .then(res => {
          if (!res.ok) {
            return res.json().then(err => {
              throw new Error(err.details || err.error || t('systemAdminTab.failedToFetchDevices', 'Failed to fetch devices'));
            });
          }
          return res.json();
        })
        .then(data => setDevices(data.devices || []))
        .catch(err => {
          console.error('Error fetching devices:', err);
          setError(err.message || t('systemAdminTab.failedToFetchDevices', 'Failed to fetch devices'));
          setDevices([]);
        });
    }
  }, [activeTab, t, userRoleId]);

  const handleRefreshSystem = async () => {
    setLoading(true);
    setError(null);
    setSuccess(null);
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      setSuccess(t('systemAdminTab.systemRefreshed', 'System refreshed successfully'));
    } catch (err) {
      setError(t('systemAdminTab.failedToRefresh', 'Failed to refresh system'));
      console.error('Error refreshing system:', err);
    } finally {
      setLoading(false);
    }
  };

  const handleSendNotification = async (e: React.FormEvent) => {
    e.preventDefault();
    setNotifLoading(true);
    setNotifError(null);
    setNotifSuccess(null);
    try {
      const res = await fetch('/api/push/send', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          to: selectedUser,
          title: notifTitle,
          message: notifMessage,
        })
      });
      if (!res.ok) {
        const errData = await res.json();
        throw new Error(errData.error || t('systemAdminTab.failedToSendNotification', 'Failed to send notification'));
      }
      setNotifSuccess(t('systemAdminTab.notificationSent', 'Notification sent successfully!'));
      setNotifTitle('');
      setNotifMessage('');
      setSelectedUser('all');
    } catch (err: any) {
      setNotifError(err.message || t('systemAdminTab.failedToSendNotification', 'Failed to send notification'));
    } finally {
      setNotifLoading(false);
    }
  };

  const handleRemoveDevice = async (deviceId: number) => {
    if (!window.confirm(t('systemAdminTab.confirmRemoveDevice', 'Are you sure you want to remove this device?'))) return;
    try {
      const res = await fetch(`/api/postgres/delete-user-device`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ id: deviceId })
      });
      if (!res.ok) throw new Error(t('systemAdminTab.failedToRemoveDevice', 'Failed to remove device'));
      setSuccess(t('systemAdminTab.deviceRemoved', 'Device removed successfully'));
      const endpoint = userRoleId === 0 ? '/api/postgres/all-devices' : '/api/postgres/all-user-devices';
      fetch(endpoint)
        .then(res => res.json())
        .then(data => setDevices(data.devices || []))
        .catch(() => setDevices([]));
    } catch (err) {
      setError(t('systemAdminTab.failedToRemoveDevice', 'Failed to remove device'));
    }
  };

  const uniqueUsers = Array.from(new Map(users.map(u => [u.email, u])).values());

  // Массив табов для выпадающего меню и табов
  const tabOptions = React.useMemo(() => {
    if (userRoleId === 1) {
      return [
        { value: 'push', label: t('systemAdminTab.sendPushNotification', 'Send Push Notification') },
        { value: 'payment-history', label: t('paymentHistory.title', 'Payment History') },
        { value: 'user-settings', label: t('settings.title', 'User Settings') },
        { value: 'pending-orders', label: t('orders.pendingOrders', 'Pending Orders') },
      ];
    }
    return [
    { value: 'config', label: t('systemAdminTab.systemConfiguration', 'System Configuration') },
    { value: 'push', label: t('systemAdminTab.sendPushNotification', 'Send Push Notification') },
    { value: 'devices', label: t('systemAdminTab.deviceManagement', 'Device Management') },
    { value: 'payment-history', label: t('paymentHistory.title', 'Payment History') },
    { value: 'user-settings', label: t('settings.title', 'User Settings') },
    { value: 'pending-orders', label: t('orders.pendingOrders', 'Pending Orders') },
    ...(userPermissions?.can_manage_users && userRoleId === 0 ? [
      { value: 'roles', label: t('systemAdminTab.rolePermissions', 'Role Permissions') },
      { value: 'approvals', label: t('systemAdminTab.pendingApprovals', 'Pending Approvals') },
    ] : [])
  ];
  }, [t, userPermissions, userRoleId]);

  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  // Фильтрация устройств по поиску
  const filteredMobileDevices = React.useMemo(() => {
    if (!search.trim()) return devices;
    const s = search.trim().toLowerCase();
    return devices.filter(device =>
      (device.device_name && device.device_name.toLowerCase().includes(s)) ||
      (device.device_type && device.device_type.toLowerCase().includes(s)) ||
      (device.user_id && String(device.user_id).toLowerCase().includes(s))
    );
  }, [devices, search]);

  return (
    <div className={`space-y-6 ${isRTL ? 'rtl' : ''}`} dir={isRTL ? 'rtl' : 'ltr'} style={window.innerWidth === 768 ? { paddingBottom: 96 } : window.innerWidth < 768 ? { paddingBottom: 96 } : {}}>
      {/* Навигация: мобильное меню или десктопные табы */}
      {isMobile ? (
        <div style={{ position: 'relative', minHeight: 56 }}>
          <div className="absolute z-30" style={{ top: 24, [isRTL ? 'right' : 'left']: 0 }}>
            <button
              type="button"
              className="p-2 border rounded shadow bg-white"
              style={{ background: '#fff' }}
              onClick={() => setMobileMenuOpen(v => !v)}
              aria-label={t('systemAdminTab.openMenu', 'Open menu')}
            >
              <svg width="24" height="24" fill="#fff" stroke="#2a4161" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><line x1="3" y1="12" x2="21" y2="12"/><line x1="3" y1="6" x2="21" y2="6"/><line x1="3" y1="18" x2="21" y2="18"/></svg>
            </button>
            {mobileMenuOpen && (
              <div className="absolute z-40 border rounded shadow w-56 mt-2 bg-white" style={{ top: 40, [isRTL ? 'right' : 'left']: 0 }}>
                {/* Активный таб сверху, остальные ниже */}
                {(() => {
                  const tabs = isRTL ? [...tabOptions].reverse() : tabOptions;
                  const active = tabs.find(tab => tab.value === activeTab);
                  const rest = tabs.filter(tab => tab.value !== activeTab);
                  return [
                    active && (
                      <button
                        key={active.value}
                        className={`block w-full text-right px-4 py-2 font-bold`}
                        style={{ background: '#accaf0' }}
                        onClick={() => { setActiveTab(active.value as any); setMobileMenuOpen(false); }}
                        disabled
                      >
                        {active.label}
                      </button>
                    ),
                    ...rest.map(tab => (
                      <button
                        key={tab.value}
                        className={`block w-full text-right px-4 py-2 hover:bg-muted`}
                        onClick={() => { setActiveTab(tab.value as any); setMobileMenuOpen(false); }}
                      >
                        {tab.label}
                      </button>
                    ))
                  ];
                })()}
              </div>
            )}
          </div>
        </div>
      ) : (
        <div className={`flex flex-nowrap overflow-x-auto gap-2 mb-4 ${isRTL ? 'flex-row-reverse' : 'flex-row'} w-full`} style={window.innerWidth === 768 ? { marginBottom: 80 } : {}}>
          {(isRTL ? [...tabOptions].reverse() : tabOptions).map(tab => (
            <Button
              key={tab.value}
              variant={activeTab === tab.value ? 'default' : 'outline'}
              onClick={() => setActiveTab(tab.value as any)}
              className={`flex-1 min-w-[180px]${activeTab === tab.value && window.innerWidth === 768 ? ' mb-20' : ''}`}
              style={activeTab === tab.value ? { background: '#accaf0', fontWeight: 'bold' } : {}}
            >
              {tab.label}
            </Button>
          ))}
        </div>
      )}

      {/* Tab Content */}
      {activeTab === 'config' && (userRoleId !== 1 ? (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <SystemConfigurationCard />
        </div>
      ) : (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <SystemStatsCard />
        </div>
      ))}
      {activeTab === 'devices' && userRoleId !== 1 && (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <DeviceManagementTable />
        </div>
      )}
      {activeTab === 'push' && (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <Card>
            <CardHeader>
              <CardTitle className={isRTL ? 'text-right' : ''}>{t('systemAdminTab.sendPushNotification', 'Send Push Notification')}</CardTitle>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSendNotification} className="space-y-4 max-w-md w-full mx-auto">
                <div>
                  <label className={`block mb-1 font-medium ${isRTL ? 'text-right' : ''}`}>{t('systemAdminTab.sendTo', 'Send To')}</label>
                  {/* Кастомный select */}
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <button
                        type="button"
                        className={`w-full border rounded p-2 mb-2 bg-white ${isRTL ? 'text-right' : 'text-left'}`}
                        style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                      >
                        {selectedUser === 'all'
                          ? t('systemAdminTab.allUsers', 'All Users')
                          : (
                              uniqueUsers.find(u => u.email === selectedUser)?.f_name || ''
                            ) +
                            (uniqueUsers.find(u => u.email === selectedUser)?.s_name
                              ? ' ' + uniqueUsers.find(u => u.email === selectedUser)?.s_name
                              : '') || selectedUser}
                      </button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent className="w-full max-h-[35vh] overflow-y-auto p-2 z-50">
                      <input
                        type="text"
                        placeholder={t('systemAdminTab.searchUser', 'Search user...')}
                        className="w-full border rounded p-2 mb-2"
                        value={searchUser || ''}
                        onChange={e => setSearchUser(e.target.value)}
                        dir={isRTL ? 'rtl' : 'ltr'}
                        style={{ textAlign: isRTL ? 'right' : 'left' }}
                      />
                      <DropdownMenuItem
                        onSelect={() => setSelectedUser('all')}
                        className={selectedUser === 'all' ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                        style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                      >
                        {t('systemAdminTab.allUsers', 'All Users')}
                      </DropdownMenuItem>
                      {uniqueUsers
                        .filter(u => {
                          const search = (searchUser || '').toLowerCase();
                          return (
                            (u.f_name && u.f_name.toLowerCase().includes(search)) ||
                            (u.s_name && u.s_name.toLowerCase().includes(search))
                          );
                        })
                        .map(u => (
                          <DropdownMenuItem
                            key={u.email}
                            onSelect={() => setSelectedUser(u.email)}
                            className={selectedUser === u.email ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                            style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                          >
                            {(u.f_name || '') + (u.s_name ? ' ' + u.s_name : '') || u.email}
                          </DropdownMenuItem>
                        ))}
                    </DropdownMenuContent>
                  </DropdownMenu>
                </div>
                <div>
                  <label className={`block mb-1 font-medium ${isRTL ? 'text-right' : ''}`}>{t('systemAdminTab.title', 'Title')}</label>
                  <input
                    className="w-full border rounded p-2"
                    value={notifTitle}
                    onChange={e => setNotifTitle(e.target.value)}
                    required
                    dir={isRTL ? 'rtl' : 'ltr'}
                  />
                </div>
                <div>
                  <label className={`block mb-1 font-medium ${isRTL ? 'text-right' : ''}`}>{t('systemAdminTab.message', 'Message')}</label>
                  <textarea
                    className="w-full border rounded p-2"
                    value={notifMessage}
                    onChange={e => setNotifMessage(e.target.value)}
                    required
                    dir={isRTL ? 'rtl' : 'ltr'}
                  />
                </div>
                <Button type="submit" disabled={notifLoading} className="w-full" style={{ background: '#0B674E', color: '#fff' }}>
                  {notifLoading ? t('systemAdminTab.sending', 'Sending...') : t('systemAdminTab.sendNotification', 'Send Notification')}
                </Button>
                {notifError && (
                  <Alert variant="destructive">
                    <AlertCircle className="h-4 w-4" />
                    <AlertTitle>{t('common.error', 'Error')}</AlertTitle>
                    <AlertDescription>{notifError}</AlertDescription>
                  </Alert>
                )}
                {notifSuccess && (
                  <Alert>
                    <AlertTitle>{t('common.success', 'Success')}</AlertTitle>
                    <AlertDescription>{notifSuccess}</AlertDescription>
                  </Alert>
                )}
              </form>
            </CardContent>
          </Card>
        </div>
      )}
      {activeTab === 'roles' && userPermissions?.can_manage_users && userRoleId === 0 && (
        <RoleManagement />
      )}
      {activeTab === 'approvals' && userPermissions?.can_manage_users && userRoleId === 0 && (
        <PendingApprovals />
      )}
      {activeTab === 'payment-history' && (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <PaymentHistoryTab />
        </div>
      )}
      {activeTab === 'user-settings' && (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <UserSettingsTab />
        </div>
      )}
      {activeTab === 'pending-orders' && (
        <div style={window.innerWidth === 768 ? { marginBottom: 80, width: '90vw', maxWidth: '90vw' } : isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
          <PendingOrdersTab />
        </div>
      )}
      {error && (
        <Alert variant="destructive">
          <AlertCircle className="h-4 w-4" />
          <AlertTitle>{t('common.error', 'Error')}</AlertTitle>
          <AlertDescription>{error}</AlertDescription>
        </Alert>
      )}
      {success && (
        <Alert>
          <AlertTitle>{t('common.success', 'Success')}</AlertTitle>
          <AlertDescription>{success}</AlertDescription>
        </Alert>
      )}
    </div>
  );
};

export default SystemAdminTab; 