import { NextRequest, NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function GET(request: Request) {
  const { searchParams } = new URL(request.url);
  const p_id = searchParams.get('p_id');

  if (!p_id) {
    return NextResponse.json({ error: 'Missing p_id parameter' }, { status: 400 });
  }

  try {
    // Получаем все связи для пользователя
    const query = `
      SELECT 
        id,
        main_user_p_id,
        related_user_p_id,
        relationship_type,
        created_at
      FROM user_relationships
      WHERE main_user_p_id = $1 OR related_user_p_id = $1
      ORDER BY created_at DESC
    `;
    
    const result = await postgresClient.query(query, [p_id]);
    
    // Если нет записей, вернуть пустой массив
    if (!result.rows || result.rows.length === 0) {
      return NextResponse.json({ relationships: [], count: 0 });
    }
    
    return NextResponse.json({ 
      relationships: result.rows,
      count: result.rows.length
    });
  } catch (error) {
    console.error('Failed to fetch user relationships:', error);
    
    // Если таблица не существует, вернуть пустой массив
    if (error instanceof Error && (
      error.message.includes('relation') && error.message.includes('does not exist') ||
      error.message.includes('column') && error.message.includes('does not exist')
    )) {
      return NextResponse.json({ 
        relationships: [],
        count: 0
      });
    }
    
    return NextResponse.json({ 
      error: 'Failed to fetch user relationships', 
      details: error instanceof Error ? error.message : 'Unknown error' 
    }, { status: 500 });
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { main_user_p_id, related_user_p_id, relationship_type = 'spouse' } = body;

    if (!main_user_p_id || !related_user_p_id) {
      return NextResponse.json({ error: 'Missing main_user_p_id or related_user_p_id parameter' }, { status: 400 });
    }

    if (main_user_p_id === related_user_p_id) {
      return NextResponse.json({ error: 'Cannot create relationship with yourself' }, { status: 400 });
    }

    // Проверяем, не существует ли уже связь
    const checkQuery = `
      SELECT id FROM user_relationships 
      WHERE (main_user_p_id = $1 AND related_user_p_id = $2) 
         OR (main_user_p_id = $2 AND related_user_p_id = $1)
    `;
    const checkResult = await postgresClient.query(checkQuery, [main_user_p_id, related_user_p_id]);
    
    if (checkResult.rows.length > 0) {
      return NextResponse.json({ 
        error: 'Relationship already exists',
        details: 'These users are already linked'
      }, { status: 409 });
    }

    // Создаем связь
    const query = `
      INSERT INTO user_relationships (main_user_p_id, related_user_p_id, relationship_type, created_at)
      VALUES ($1, $2, $3, NOW())
      RETURNING id, main_user_p_id, related_user_p_id, relationship_type, created_at
    `;
    
    const values = [main_user_p_id, related_user_p_id, relationship_type];
    const result = await postgresClient.query(query, values);
    
    return NextResponse.json({ 
      success: true,
      relationship: result.rows[0]
    });
  } catch (error) {
    console.error('Failed to create user relationship:', error);
    
    // Если таблица не существует, вернуть ошибку
    if (error instanceof Error && (
      error.message.includes('relation') && error.message.includes('does not exist') ||
      error.message.includes('column') && error.message.includes('does not exist')
    )) {
      return NextResponse.json({ 
        error: 'User relationships table does not exist',
        details: error.message
      }, { status: 500 });
    }
    
    return NextResponse.json({ 
      error: 'Failed to create user relationship', 
      details: error instanceof Error ? error.message : 'Unknown error' 
    }, { status: 500 });
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const body = await request.json();
    const { main_user_p_id, related_user_p_id } = body;

    if (!main_user_p_id || !related_user_p_id) {
      return NextResponse.json({ error: 'Missing main_user_p_id or related_user_p_id parameter' }, { status: 400 });
    }

    // Удаляем связь
    const query = `
      DELETE FROM user_relationships 
      WHERE (main_user_p_id = $1 AND related_user_p_id = $2) 
         OR (main_user_p_id = $2 AND related_user_p_id = $1)
      RETURNING id
    `;
    
    const result = await postgresClient.query(query, [main_user_p_id, related_user_p_id]);
    
    if (result.rows.length === 0) {
      return NextResponse.json({ 
        error: 'Relationship not found',
        details: 'No relationship exists between these users'
      }, { status: 404 });
    }
    
    return NextResponse.json({ 
      success: true,
      message: 'Relationship deleted successfully'
    });
  } catch (error) {
    console.error('Failed to delete user relationship:', error);
    
    return NextResponse.json({ 
      error: 'Failed to delete user relationship', 
      details: error instanceof Error ? error.message : 'Unknown error' 
    }, { status: 500 });
  }
} 