import { NextResponse } from 'next/server';
import { pool } from '../../../../lib/db';

export async function GET(request: Request) {
  try {
    const { searchParams } = new URL(request.url);
    const email = searchParams.get('email');
    const p_id = searchParams.get('p_id');
    const allUserIdsForEmail = searchParams.get('all_user_ids_for_email');
    
    let result;
    
    if (email) {
      // Возвращаем детерминированно "основного" пользователя по данному email
      // Выбираем самого раннего по user_id (минимальный user_id)
      result = await pool.query(
        'SELECT user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id, payment_methods, order_tab FROM users WHERE email = $1 ORDER BY user_id ASC',
        [email]
      );
      if (result.rows.length === 0) {
        return NextResponse.json({ users: [] });
      }
      return NextResponse.json(result.rows[0]);
    } else if (p_id) {
      result = await pool.query(
        'SELECT user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id, payment_methods, order_tab FROM users WHERE p_id = $1',
        [p_id]
      );
      if (result.rows.length === 0) {
        return NextResponse.json({ users: [] });
      }
      return NextResponse.json(result.rows[0]);
    } else if (allUserIdsForEmail) {
      // Также возвращаем все user_id в детерминированном порядке
      result = await pool.query(
        'SELECT user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id, payment_methods, order_tab FROM users WHERE email = $1 ORDER BY user_id ASC',
        [allUserIdsForEmail]
      );
      return NextResponse.json({ users: result.rows });
    } else {
      result = await pool.query(
        'SELECT user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id, payment_methods, order_tab FROM users'
      );
      return NextResponse.json({ users: result.rows });
    }
  } catch (error: any) {
    console.error('Error fetching users:', error);
    return NextResponse.json(
      { error: 'Failed to fetch users' },
      { status: 500 }
    );
  }
}

export async function PUT(request: Request) {
  try {
    const body = await request.json();
    const { email, userId, ...updateData } = body;
    
    if (!email && !userId) {
      return NextResponse.json(
        { error: 'Email or userId is required' },
        { status: 400 }
      );
    }

    let query;
    let values;
    
    // Если payment_methods — массив, сериализуем в JSON
    if (updateData.payment_methods && Array.isArray(updateData.payment_methods)) {
      updateData.payment_methods = JSON.stringify(updateData.payment_methods);
    }
    
    if (userId) {
      // Обновление по user_id
      const fields = Object.keys(updateData);
      if (fields.length === 0) {
        return NextResponse.json(
          { error: 'No fields to update' },
          { status: 400 }
        );
      }
      
      query = `UPDATE users SET ${fields.map((field, index) => `${field} = $${index + 1}`).join(', ')} WHERE user_id = $${fields.length + 1} RETURNING user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id, payment_methods, order_tab`;
      values = [...Object.values(updateData), userId];
    } else {
      // Обновление по email
      const fields = Object.keys(updateData);
      if (fields.length === 0) {
        return NextResponse.json(
          { error: 'No fields to update' },
          { status: 400 }
        );
      }
      
      query = `UPDATE users SET ${fields.map((field, index) => `${field} = $${index + 1}`).join(', ')} WHERE email = $${fields.length + 1} RETURNING user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id, payment_methods, order_tab`;
      values = [...Object.values(updateData), email];
    }

    const result = await pool.query(query, values);

    if (result.rows.length === 0) {
      return NextResponse.json(
        { error: 'User not found' },
        { status: 404 }
      );
    }

    return NextResponse.json({ user: result.rows[0] });
  } catch (error: any) {
    console.error('Error updating user:', error);
    return NextResponse.json(
      { error: 'Failed to update user' },
      { status: 500 }
    );
  }
}

export async function POST(request: Request) {
  try {
    const body = await request.json();
    const { 
      f_name, 
      s_name, 
      email, 
      phone, 
      role_id = 2, 
      device_id, 
      device_ids, 
      main_user_p_id, 
      relationship_type = 'spouse',
      p_id
    } = body;

    if (!f_name || !s_name || !email || !p_id) {
      return NextResponse.json(
        { error: 'f_name, s_name, email, and p_id are required' },
        { status: 400 }
      );
    }

    // Проверяем, не существует ли уже пользователь с таким email
    const existingUser = await pool.query(
      'SELECT user_id FROM users WHERE email = $1',
      [email]
    );

    if (existingUser.rows.length > 0) {
      return NextResponse.json(
        { error: 'User with this email already exists' },
        { status: 409 }
      );
    }

    // Используем p_id, переданный пользователем
    const new_p_id = p_id;

    // Хелпер создания пользователя для одного device_id
    const createOneUser = async (singleDeviceId: string | null) => {
      const emailPrefix = email.split('@')[0];
      const user_name = singleDeviceId ? `${emailPrefix}_${singleDeviceId}` : emailPrefix;

      // Формируем uname из f_name и s_name
      const uname = `${f_name} ${s_name}`.trim();

      // Получаем num_of_members из пользователя по singleDeviceId
      let num_of_members = 2; // По умолчанию 2
      if (singleDeviceId) {
        try {
          const existingUserQuery = `
            SELECT num_of_members FROM users 
            WHERE device_id = $1 
            LIMIT 1
          `;
          const existingUserResult = await pool.query(existingUserQuery, [singleDeviceId]);
          if (existingUserResult.rows.length > 0) {
            num_of_members = existingUserResult.rows[0].num_of_members || 2;
          }
        } catch (error) {
          console.warn('Failed to get num_of_members from existing user:', error);
          num_of_members = 2; // Fallback значение
        }
      }

      // Генерируем временный пароль (пользователь сможет изменить его позже)
      const tempPassword = `temp_${Date.now()}`;

      // Создаем нового пользователя
      const insertQuery = `
        INSERT INTO users (f_name, s_name, email, phone, role_id, device_id, p_id, user_name, password, uname, num_of_members)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
        RETURNING user_id, user_name, f_name, s_name, p_id, email, phone, role_id, device_id, uid, payment_history_id
      `;

      const insertValues = [f_name, s_name, email, phone || null, role_id, singleDeviceId || null, new_p_id, user_name, tempPassword, uname, num_of_members];
      const result = await pool.query(insertQuery, insertValues);

      // Создаем связь в таблице user_relationships
      if (singleDeviceId) {
        try {
          // Получаем p_id существующего пользователя с тем же device_id
          const existingUserQuery = `
            SELECT p_id FROM users 
            WHERE device_id = $1 AND p_id != $2
            LIMIT 1
          `;
          const existingUserResult = await pool.query(existingUserQuery, [singleDeviceId, new_p_id]);
          
          if (existingUserResult.rows.length > 0) {
            const main_user_p_id = existingUserResult.rows[0].p_id;
            
            // Создаем связь между пользователями
            const relationshipQuery = `
              INSERT INTO user_relationships (main_user_p_id, related_user_p_id, relationship_type, created_at)
              VALUES ($1, $2, $3, NOW())
            `;
            
            await pool.query(relationshipQuery, [main_user_p_id, new_p_id, body.relationship || 'family_member']);
          }
        } catch (error) {
          console.error('Failed to create user relationship:', error);
          // Продолжаем выполнение, даже если не удалось создать связь
        }
      }

      // Создаем запись в таблице user_meters
      if (singleDeviceId && result.rows[0].user_id) {
        try {
          const userMeterQuery = `
            INSERT INTO user_meters (user_id, device_id)
            VALUES ($1, $2)
          `;
          
          await pool.query(userMeterQuery, [result.rows[0].user_id, singleDeviceId]);
        } catch (error) {
          console.error('Failed to create user_meters record:', error);
          // Продолжаем выполнение, даже если не удалось создать запись в user_meters
        }
      }

      return result.rows[0];
    };

    // Если пришел массив device_ids — создаем по каждому устройству отдельного пользователя
    if (Array.isArray(device_ids) && device_ids.length > 0) {
      const createdUsers = [] as any[];
      for (const did of device_ids) {
        const u = await createOneUser(did);
        createdUsers.push(u);
      }
      return NextResponse.json({ success: true, users: createdUsers });
    }

    // Иначе — одиночный сценарий (как раньше)
    const singleUser = await createOneUser(device_id || null);

    // Если указан main_user_p_id (не админ), создаем связь между пользователями
    if (main_user_p_id && main_user_p_id !== new_p_id) {
      try {
        const relationshipResponse = await fetch(`${request.url.split('/api/')[0]}/api/postgres/user-relationships`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            main_user_p_id: parseInt(main_user_p_id),
            related_user_p_id: new_p_id,
            relationship_type
          })
        });

        if (!relationshipResponse.ok) {
          console.error('Failed to create user relationship:', await relationshipResponse.text());
        }
      } catch (error) {
        console.error('Error creating user relationship:', error);
      }
    }

    return NextResponse.json({ 
      success: true,
      user: singleUser
    });
  } catch (error: any) {
    console.error('Error creating user:', error);
    return NextResponse.json(
      { error: 'Failed to create user' },
      { status: 500 }
    );
  }
} 