"use client";

import React, { useEffect, useState } from 'react';
import { useTranslation } from 'react-i18next';
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from "../components/ui/card";
import { useAuth } from '../hooks/use-auth';
import { format } from 'date-fns';

const API_KEY = process.env.NEXT_PUBLIC_SUMIT_TOKEN;
const COMPANY_ID = process.env.NEXT_PUBLIC_SUMIT_COMPANY_ID;

interface ItemRow {
  DocumentID: string;
  Date: string;
  Name: string;
  Quantity: number;
  UnitPrice: number;
  TotalPrice: number;
}

const PaymentHistoryTab: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const { user } = useAuth();
  const [pId, setPId] = useState<string | null>(null);
  const [dateFrom, setDateFrom] = useState<string>('2023-01-01');
  const [dateTo, setDateTo] = useState<string>(format(new Date(), 'yyyy-MM-dd'));
  const [items, setItems] = useState<ItemRow[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [openDocId, setOpenDocId] = useState<string | null>(null);
  const [docDetails, setDocDetails] = useState<Record<string, { items: ItemRow[], downloadUrl?: string }>>({});

  // Получить p_id текущего пользователя
  useEffect(() => {
    if (user?.email) {
      fetch(`/api/postgres/users?email=${encodeURIComponent(user.email)}`)
        .then(res => res.json())
        .then(data => setPId(data.p_id || null));
    }
  }, [user]);

  // Загрузка истории платежей по p_id и датам
  useEffect(() => {
    if (!pId || !dateFrom || !dateTo) {
      setItems([]);
      return;
    }
    setLoading(true);
    setError(null);
    async function fetchAllDocuments() {
      let allDocs: any[] = [];
      let page = 0;
      const pageSize = 100;
      while (true) {
        const resp = await fetch('https://api.sumit.co.il/accounting/documents/list/', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            Accept: 'application/json',
          },
          body: JSON.stringify({
            Credentials: {
              CompanyID: COMPANY_ID,
              APIKey: API_KEY,
            },
            DateFrom: dateFrom + 'T00:00:00Z',
            DateTo: dateTo + 'T23:59:59Z',
            IncludeDrafts: false,
            Paging: {
              StartIndex: page * pageSize,
              PageSize: pageSize,
            },
          }),
        });
        const json = await resp.json();
        const docs = json?.Data?.Documents || [];
        allDocs = allDocs.concat(docs);
        if (docs.length < pageSize) break;
        page++;
      }
      return allDocs;
    }
    async function fetchData() {
      try {
        // 1. Получить все документы с пагинацией
        const allDocs = (await fetchAllDocuments()) || [];
        // 2. Фильтруем по CustomerName (p_id)
        const docs = allDocs.filter((doc: any) => doc.CustomerName === pId);
        // 3. Для каждого документа — детали
        const allItems: ItemRow[] = [];
        for (const doc of docs) {
          const res = await fetch('https://api.sumit.co.il/accounting/documents/getdetails/', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              Accept: 'application/json',
            },
            body: JSON.stringify({
              Credentials: {
                CompanyID: COMPANY_ID,
                APIKey: API_KEY,
              },
              DocumentID: doc.DocumentID,
              DocumentType: "0",
            }),
          });
          const details = await res.json();
          const itemsArr = details?.Data?.Items || [];
          const docDate = details?.Data?.Document?.Date || '';
          for (const item of itemsArr) {
            allItems.push({
              DocumentID: doc.DocumentID,
              Date: docDate,
              Name: item.Item?.Name || '',
              Quantity: item.Quantity,
              UnitPrice: item.UnitPrice,
              TotalPrice: item.TotalPrice,
            });
          }
        }
        setItems(allItems);
      } catch (e: any) {
        setError(e.message || t('paymentHistory.error', 'Ошибка загрузки данных'));
        setItems([]);
      }
      setLoading(false);
    }
    fetchData();
  }, [pId, dateFrom, dateTo, t]);

  // Группировка по DocumentID
  const groupedDocs = items.reduce<Record<string, { date: string, total: number, items: ItemRow[] }>>((acc, item) => {
    if (!acc[item.DocumentID]) {
      acc[item.DocumentID] = { date: item.Date, total: 0, items: [] };
    }
    acc[item.DocumentID].total += item.TotalPrice;
    acc[item.DocumentID].items.push(item);
    return acc;
  }, {});
  const docIds = Object.keys(groupedDocs);

  // Функция для загрузки деталей документа (если не загружено)
  async function handleOpenDoc(docId: string) {
    if (docDetails[docId]) {
      setOpenDocId(openDocId === docId ? null : docId);
      return;
    }
    setLoading(true);
    try {
      const res = await fetch('https://api.sumit.co.il/accounting/documents/getdetails/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          Accept: 'application/json',
        },
        body: JSON.stringify({
          Credentials: {
            CompanyID: COMPANY_ID,
            APIKey: API_KEY,
          },
          DocumentID: docId,
          DocumentType: "0",
        }),
      });
      const details = await res.json();
      const itemsArr = details?.Data?.Items || [];
      // Корректно извлекаем DocumentDownloadURL
      const downloadUrl = details?.Data?.DocumentDownloadURL || '';
      setDocDetails(prev => ({ ...prev, [docId]: { items: itemsArr, downloadUrl } }));
      setOpenDocId(docId);
    } catch (e) {
      setError(t('paymentHistory.detailsError', 'Ошибка загрузки деталей документа'));
    }
    setLoading(false);
  }

  return (
    <Card className="w-full max-w-2xl mx-auto sm:max-w-full">
      <CardHeader>
        <CardTitle className="flex items-center text-lg sm:text-base">
          {t('paymentHistory.title', 'История платежей')}
        </CardTitle>
        <CardDescription className="sm:text-xs">{t('paymentHistory.description', 'Просмотрите историю ваших платежей ниже.')}</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="mb-4 flex flex-col sm:flex-row gap-2 items-center">
          {/* Выбор периода */}
          <div className="flex gap-2 items-center">
            <label className="font-medium text-sm">{t('paymentHistory.dateFrom', 'С')}</label>
            <input type="date" value={dateFrom} onChange={e => setDateFrom(e.target.value)} className="border rounded p-1" />
            <label className="font-medium text-sm">{t('paymentHistory.dateTo', 'По')}</label>
            <input type="date" value={dateTo} onChange={e => setDateTo(e.target.value)} className="border rounded p-1" />
          </div>
        </div>
        {/* Таблица с результатами */}
        {loading ? (
          <p>{t('paymentHistory.loading', 'Загрузка истории платежей...')}</p>
        ) : error ? (
          <p className="text-red-500">{error}</p>
        ) : docIds.length > 0 ? (
          <div className="overflow-x-auto max-h-[60vh] overflow-y-auto">
            <table className="min-w-full border text-sm">
              <thead>
                <tr style={{ background: '#e5e5e5' }}>
                  <th className="border px-2 py-1">{t('paymentHistory.documentId', 'Документ')}</th>
                  <th className="border px-2 py-1">{t('paymentHistory.date', 'Дата')}</th>
                  <th className="border px-2 py-1">{t('paymentHistory.totalPrice', 'Сумма')}</th>
                </tr>
              </thead>
              <tbody>
                {docIds.map(docId => (
                  <React.Fragment key={docId}>
                    <tr
                      className={`cursor-pointer hover:bg-accent ${openDocId === docId ? 'bg-[#f0f4fa]' : ''}`}
                      onClick={() => handleOpenDoc(docId)}
                    >
                      <td className="border px-2 py-1">{docId}</td>
                      <td className="border px-2 py-1">{groupedDocs[docId].date ? format(new Date(groupedDocs[docId].date), 'yyyy-MM-dd') : ''}</td>
                      <td className="border px-2 py-1">{groupedDocs[docId].total.toFixed(2)}</td>
                    </tr>
                    {openDocId === docId && docDetails[docId] && (
                      <tr>
                        <td colSpan={3} className="p-0">
                          <div className="p-2 border-t bg-white">
                            <div className="mb-2 font-medium">{t('paymentHistory.items', 'Позиции')}</div>
                            <table className="min-w-full border text-xs mb-2">
                              <thead>
                                <tr style={{ background: '#e5e5e5' }}>
                                  <th className="border px-2 py-1">{t('paymentHistory.item', 'Товар')}</th>
                                  <th className="border px-2 py-1">{t('paymentHistory.quantity', 'Кол-во')}</th>
                                  <th className="border px-2 py-1">{t('paymentHistory.unitPrice', 'Цена за ед.')}</th>
                                  <th className="border px-2 py-1">{t('paymentHistory.totalPrice', 'Сумма')}</th>
                                </tr>
                              </thead>
                              <tbody>
                                {docDetails[docId].items.map((item, idx) => (
                                  <tr key={idx}>
                                    <td className="border px-2 py-1">{(item as any).Name ?? (item as any).Item?.Name ?? ''}</td>
                                    <td className="border px-2 py-1">{item.Quantity}</td>
                                    <td className="border px-2 py-1">{item.UnitPrice}</td>
                                    <td className="border px-2 py-1">{item.TotalPrice}</td>
                                  </tr>
                                ))}
                              </tbody>
                            </table>
                            {/* Кнопка для скачивания документа */}
                            {docDetails[docId].downloadUrl && (
                              <a
                                href={docDetails[docId].downloadUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-block px-4 py-2 bg-[#0B674E] text-white rounded hover:bg-[#09553d] mt-2"
                              >
                                {t('paymentHistory.download', 'Скачать документ')}
                              </a>
                            )}
                          </div>
                        </td>
                      </tr>
                    )}
                  </React.Fragment>
                ))}
              </tbody>
            </table>
          </div>
        ) : pId ? (
          <p>{t('paymentHistory.noData', 'Нет данных за выбранный период.')}</p>
        ) : null}
      </CardContent>
    </Card>
  );
};

export default PaymentHistoryTab;